<?php
/**
 * WooCommerce hook handlers.
 *
 * Listens for order events and sends notifications to the GotASale server.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class GotASale_WooCommerce {

	/**
	 * Register WooCommerce hooks.
	 */
	public function init() {
		add_action( 'woocommerce_checkout_order_processed', array( $this, 'handle_new_order' ), 10, 2 );
		add_action( 'woocommerce_payment_complete', array( $this, 'handle_payment_complete' ) );
		add_action( 'woocommerce_order_status_changed', array( $this, 'handle_status_changed' ), 10, 4 );
		add_action( 'woocommerce_low_stock', array( $this, 'handle_low_stock' ) );
		add_action( 'woocommerce_no_stock', array( $this, 'handle_low_stock' ) );
		add_action( 'woocommerce_order_refunded', array( $this, 'handle_refund_created' ), 10, 2 );
		add_action( 'woocommerce_order_status_failed', array( $this, 'handle_payment_failed' ), 10, 2 );
		add_action( 'woocommerce_order_status_cancelled', array( $this, 'handle_order_cancelled' ), 10, 2 );
	}

	/**
	 * Handle new order creation (via checkout).
	 *
	 * Fires on woocommerce_checkout_order_processed — after items are saved.
	 * Pending orders are deferred to handle_payment_complete() so redirect-based
	 * gateways (PayPal, Afterpay, Klarna) only notify once payment confirms.
	 *
	 * @param int      $order_id Order ID.
	 * @param WC_Order $order    Order object.
	 */
	public function handle_new_order( $order_id, $order = null ) {
		$dedup_key = 'gotasale_sent_' . md5( 'new_order_' . $order_id );
		if ( get_transient( $dedup_key ) ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		// Pending = redirect-based gateway (PayPal, Afterpay, Klarna).
		// Wait for woocommerce_payment_complete to fire with confirmed payment.
		if ( $order->get_status() === 'pending' ) {
			return;
		}

		// processing/completed = instant payment, on-hold = offline payment.
		// Both should notify immediately.
		$payload = $this->format_order_payload( $order, 'new_order' );
		GotASale_API::send_notification( $payload );
		set_transient( $dedup_key, true, 5 * MINUTE_IN_SECONDS );
	}

	/**
	 * Handle payment completion for redirect-based gateways.
	 *
	 * Fires on woocommerce_payment_complete — when PayPal IPN, Afterpay,
	 * Klarna etc. confirm payment. Sends as new_order if not already sent.
	 *
	 * @param int $order_id Order ID.
	 */
	public function handle_payment_complete( $order_id ) {
		$dedup_key = 'gotasale_sent_' . md5( 'new_order_' . $order_id );
		if ( get_transient( $dedup_key ) ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		$payload = $this->format_order_payload( $order, 'new_order' );
		GotASale_API::send_notification( $payload );
		set_transient( $dedup_key, true, 5 * MINUTE_IN_SECONDS );
	}

	/**
	 * Handle order status change.
	 *
	 * @param int      $order_id   Order ID.
	 * @param string   $old_status Previous status.
	 * @param string   $new_status New status.
	 * @param WC_Order $order      Order object.
	 */
	public function handle_status_changed( $order_id, $old_status, $new_status, $order ) {
		// Order-level dedup: only 1 status_changed per order per 2 minutes.
		// Collapses rapid intermediate transitions (pending → on-hold → processing).
		$dedup_key = 'gotasale_sent_' . md5( 'status_changed_' . $order_id );
		if ( get_transient( $dedup_key ) ) {
			return;
		}

		$payload = $this->format_order_payload( $order, 'status_changed' );
		$payload['old_status'] = $old_status;
		$payload['new_status'] = $new_status;

		GotASale_API::send_notification( $payload );
		set_transient( $dedup_key, true, 2 * MINUTE_IN_SECONDS );
	}

	/**
	 * Handle low stock / no stock notification.
	 *
	 * @param WC_Product $product Product object.
	 */
	public function handle_low_stock( $product ) {
		$dedup_key = 'gotasale_sent_' . md5( 'low_stock_' . $product->get_id() );
		if ( get_transient( $dedup_key ) ) {
			return;
		}

		$payload = array(
			'event'   => 'low_stock',
			'product' => array(
				'id'           => $product->get_id(),
				'name'         => $product->get_name(),
				'sku'          => $product->get_sku(),
				'stock_quantity' => $product->get_stock_quantity(),
				'stock_status' => $product->get_stock_status(),
			),
		);

		GotASale_API::send_notification( $payload );
		set_transient( $dedup_key, true, 5 * MINUTE_IN_SECONDS );
	}

	/**
	 * Handle refund creation.
	 *
	 * @param int $order_id  Order ID.
	 * @param int $refund_id Refund ID.
	 */
	public function handle_refund_created( $order_id, $refund_id ) {
		$dedup_key = 'gotasale_sent_' . md5( 'refund_created_' . $refund_id );
		if ( get_transient( $dedup_key ) ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		$refund = wc_get_order( $refund_id );

		$payload = $this->format_order_payload( $order, 'refund_created' );
		$payload['refund_amount'] = $refund ? $refund->get_amount() : '';
		$payload['refund_reason'] = $refund ? $refund->get_reason() : '';

		GotASale_API::send_notification( $payload );
		set_transient( $dedup_key, true, MINUTE_IN_SECONDS );
	}

	/**
	 * Handle payment failed (order status changed to failed).
	 *
	 * @param int      $order_id Order ID.
	 * @param WC_Order $order    Order object.
	 */
	public function handle_payment_failed( $order_id, $order = null ) {
		$dedup_key = 'gotasale_sent_' . md5( 'payment_failed_' . $order_id );
		if ( get_transient( $dedup_key ) ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		$payload = $this->format_order_payload( $order, 'payment_failed' );
		GotASale_API::send_notification( $payload );
		set_transient( $dedup_key, true, MINUTE_IN_SECONDS );
	}

	/**
	 * Handle order cancelled.
	 *
	 * @param int      $order_id Order ID.
	 * @param WC_Order $order    Order object.
	 */
	public function handle_order_cancelled( $order_id, $order = null ) {
		$dedup_key = 'gotasale_sent_' . md5( 'order_cancelled_' . $order_id );
		if ( get_transient( $dedup_key ) ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		$payload = $this->format_order_payload( $order, 'order_cancelled' );
		GotASale_API::send_notification( $payload );
		set_transient( $dedup_key, true, MINUTE_IN_SECONDS );
	}

	/**
	 * Format order data into notification payload.
	 *
	 * @param WC_Order $order Order object.
	 * @param string   $event Event type.
	 * @return array Formatted payload.
	 */
	private function format_order_payload( $order, $event ) {
		$items = array();
		foreach ( $order->get_items() as $item ) {
			$product   = $item->get_product();
			$item_data = array(
				'name'     => $item->get_name(),
				'quantity' => $item->get_quantity(),
				'total'    => $item->get_total(),
				'sku'      => $product ? $product->get_sku() : null,
			);

			// Variation attributes (size, color, etc.)
			if ( $item->get_variation_id() ) {
				$meta = $item->get_formatted_meta_data( '_' );
				if ( ! empty( $meta ) ) {
					$item_data['meta'] = array_values( array_map( function( $m ) {
						return array( 'key' => $m->display_key, 'value' => $m->display_value );
					}, $meta ) );
				}
			}

			$items[] = $item_data;
		}

		return array(
			'event'    => $event,
			'order'    => array(
				'id'             => $order->get_id(),
				'number'         => $order->get_order_number(),
				'status'         => $order->get_status(),
				'total'          => $order->get_total(),
				'currency'       => $order->get_currency(),
				'payment_method' => $order->get_payment_method_title(),
				'date_created'   => $order->get_date_created()->format( 'c' ),
			),
			'customer' => array(
				'name'  => $order->get_formatted_billing_full_name(),
				'email' => $order->get_billing_email(),
				'phone' => $order->get_billing_phone(),
			),
			'items'    => $items,
			'pricing'  => array(
				'subtotal'       => $order->get_subtotal(),
				'shipping'       => $order->get_shipping_total(),
				'tax'            => $order->get_total_tax(),
				'total'          => $order->get_total(),
				'payment_method' => $order->get_payment_method_title(),
			),
			'address'  => array(
				'billing_country'  => $order->get_billing_country(),
				'billing_state'    => $order->get_billing_state(),
				'billing_city'     => $order->get_billing_city(),
				'shipping_country' => $order->get_shipping_country(),
				'shipping_state'   => $order->get_shipping_state(),
				'shipping_city'    => $order->get_shipping_city(),
			),
		);
	}
}
