<?php
/**
 * WooCommerce API key management.
 *
 * Creates read_write WC REST API keys programmatically and
 * sends them to the GotASale server for order proxy functionality.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class GotASale_WC_API {

	/**
	 * Set up WooCommerce API keys and send to the server.
	 * Called after successful store registration.
	 *
	 * @return bool True on success, false on failure.
	 */
	public static function setup() {
		// Skip if already configured
		if ( get_option( 'gotasale_wc_api_configured' ) ) {
			return true;
		}

		// WooCommerce must be active
		if ( ! class_exists( 'WooCommerce' ) ) {
			// Schedule a retry via WP Cron
			if ( ! wp_next_scheduled( 'gotasale_wc_api_setup_retry' ) ) {
				wp_schedule_single_event( time() + 60, 'gotasale_wc_api_setup_retry' );
			}
			return false;
		}

		global $wpdb;

		// Delete any existing GotASale keys — the consumer_key column in WC's DB
		// stores a SHA256 hash, so existing keys can never be reused for API auth.
		// Always create fresh keys to ensure we have the plain-text ck_ value.
		$wpdb->query(
			"DELETE FROM {$wpdb->prefix}woocommerce_api_keys
			 WHERE description IN ('Got A Sale - Store Manager', 'GotASale - Store Manager', 'GotASale - Order Sync')"
		);

		// Generate new API keys
		$consumer_key    = 'ck_' . wc_rand_hash();
		$consumer_secret = 'cs_' . wc_rand_hash();

		// Determine the user ID for the API key
		$api_user_id = get_current_user_id();
		if ( ! $api_user_id ) {
			// Fallback: find the first admin with manage_woocommerce capability
			$admins = get_users( array(
				'role__in' => array( 'administrator', 'shop_manager' ),
				'number'   => 1,
				'orderby'  => 'ID',
				'order'    => 'ASC',
				'fields'   => 'ID',
			) );
			$api_user_id = ! empty( $admins ) ? (int) $admins[0] : 0;
		}

		if ( ! $api_user_id ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'GotASale: No suitable user found for WC API key creation' );
			}
			return false;
		}

		$wpdb->insert(
			$wpdb->prefix . 'woocommerce_api_keys',
			array(
				'user_id'         => $api_user_id,
				'description'     => 'Got A Sale - Store Manager',
				'permissions'     => 'read_write',
				'consumer_key'    => wc_api_hash( $consumer_key ),
				'consumer_secret' => $consumer_secret,
				'truncated_key'   => substr( $consumer_key, -7 ),
			),
			array( '%d', '%s', '%s', '%s', '%s', '%s' )
		);

		if ( ! $wpdb->insert_id ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'GotASale: Failed to create WC API key' );
			}
			return false;
		}

		// Send keys to GotASale server
		$site_token = get_option( 'gotasale_site_token' );
		if ( ! $site_token ) {
			return false;
		}

		$response = wp_remote_post( GOTASALE_API_URL . '/stores/wc-credentials', array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'X-Site-Token' => $site_token,
			),
			'body'    => wp_json_encode( array(
				'consumer_key'    => $consumer_key,
				'consumer_secret' => $consumer_secret,
				'permissions'     => 'read_write',
			) ),
			'timeout' => 15,
		) );

		if ( is_wp_error( $response ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'GotASale: Failed to send WC credentials — ' . $response->get_error_message() );
			}
			// Schedule retry
			if ( ! wp_next_scheduled( 'gotasale_wc_api_setup_retry' ) ) {
				wp_schedule_single_event( time() + 120, 'gotasale_wc_api_setup_retry' );
			}
			return false;
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( 200 === $code && ! empty( $body['ok'] ) ) {
			update_option( 'gotasale_wc_api_configured', true );
			update_option( 'gotasale_wc_permissions', 'read_write' );
			return true;
		}

		$error = isset( $body['error'] ) ? $body['error'] : 'HTTP ' . $code;
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'GotASale: WC credentials rejected — ' . $error );
		}

		// Schedule retry
		if ( ! wp_next_scheduled( 'gotasale_wc_api_setup_retry' ) ) {
			wp_schedule_single_event( time() + 120, 'gotasale_wc_api_setup_retry' );
		}

		return false;
	}

	/**
	 * Regenerate and resend WC API keys.
	 * Deletes existing keys and creates fresh ones.
	 *
	 * @return bool True on success.
	 */
	public static function reconnect() {
		global $wpdb;

		// Delete existing keys (both old and new description)
		$wpdb->delete(
			$wpdb->prefix . 'woocommerce_api_keys',
			array( 'description' => 'Got A Sale - Store Manager' ),
			array( '%s' )
		);
		$wpdb->delete(
			$wpdb->prefix . 'woocommerce_api_keys',
			array( 'description' => 'GotASale - Store Manager' ),
			array( '%s' )
		);
		$wpdb->delete(
			$wpdb->prefix . 'woocommerce_api_keys',
			array( 'description' => 'GotASale - Order Sync' ),
			array( '%s' )
		);

		// Clear configured flag so setup() creates new keys
		delete_option( 'gotasale_wc_api_configured' );
		delete_option( 'gotasale_wc_permissions' );

		return self::setup();
	}

	/**
	 * Check if WC API is configured.
	 *
	 * @return bool
	 */
	public static function is_configured() {
		return (bool) get_option( 'gotasale_wc_api_configured', false );
	}
}
