<?php
/**
 * HTTP client for the GotASale API server.
 *
 * All communication with the server goes through this class.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class GotASale_API {

	/**
	 * Send a notification to the GotASale server.
	 *
	 * @param array $payload Notification data.
	 * @return array|WP_Error Response or error.
	 */
	public static function send_notification( $payload ) {
		return self::post( '/notify', $payload );
	}

	/**
	 * Send a test notification to all linked destinations.
	 *
	 * @return array|WP_Error Response or error.
	 */
	public static function send_test() {
		return self::post( '/test', array() );
	}

	/**
	 * Deactivate the license key for this store (server-side unlink).
	 *
	 * @return array|WP_Error Response or error.
	 */
	public static function deactivate_license() {
		return self::post( '/stores/deactivate', array() );
	}

	/**
	 * Get available channels for the current store.
	 *
	 * @return array|WP_Error Channels or error.
	 */
	public static function get_available_channels() {
		$response = self::get( '/available-channels' );

		// Cache response for admin UI display
		if ( ! is_wp_error( $response ) ) {
			update_option( 'gotasale_available_channels_cache', array(
				'data' => $response,
			), false );
		}

		return $response;
	}

	/**
	 * Remove a channel destination (soft-delete).
	 *
	 * @param int $id Channel link ID.
	 * @return array|WP_Error Response or error.
	 */
	public static function remove_channel( $id ) {
		return self::delete( '/available-channels/' . intval( $id ) );
	}


	/**
	 * Get pending link requests for the current store.
	 *
	 * @return array|WP_Error Pending links or error.
	 */
	public static function get_pending_links() {
		return self::get( '/stores/pending-links' );
	}

	/**
	 * Confirm a pending link request with the given code.
	 *
	 * @param string $confirm_code The 6-character confirmation code.
	 * @return array|WP_Error Response or error.
	 */
	public static function confirm_link( $confirm_code ) {
		return self::post( '/stores/confirm-link', array( 'confirm_code' => $confirm_code ) );
	}

	/**
	 * Activate a license key for this store.
	 *
	 * @param string $license_key The license key (GS-XXXX-XXXX-XXXX-XXXX).
	 * @return array|WP_Error Response or error.
	 */
	public static function activate_license( $license_key ) {
		$token = get_option( 'gotasale_site_token' );

		if ( ! $token ) {
			return new WP_Error( 'gotasale_no_token', __( 'Site token not configured.', 'got-a-sale' ) );
		}

		$response = wp_remote_post( GOTASALE_API_URL . '/stores/activate', array(
			'headers'   => array(
				'Content-Type' => 'application/json',
			),
			'body'      => wp_json_encode( array(
				'license_key' => $license_key,
				'site_token'  => $token,
			) ),
			'timeout'   => 15,
			'sslverify' => true,
		) );

		return self::parse_response( $response );
	}


	/**
	 * Create a Stripe billing portal session.
	 *
	 * @return array|WP_Error Portal URL or error.
	 */
	public static function create_billing_portal() {
		return self::post( '/billing/portal', array() );
	}

	/**
	 * Create a Stripe checkout session.
	 *
	 * @param string $tier     The tier to upgrade to (pro/agency).
	 * @param string $interval Billing interval (month/year). Default 'month'.
	 * @return array|WP_Error Checkout URL or error.
	 */
	public static function create_checkout_session( $tier, $interval = 'month' ) {
		return self::post( '/billing/checkout', array( 'tier' => $tier, 'interval' => $interval ) );
	}

	/**
	 * Parse and unwrap a server response.
	 *
	 * The server wraps all responses in { ok: true, data: {...} } or
	 * { ok: false, error: { code, message } }. This method unwraps the
	 * envelope so callers get the inner data directly, and converts
	 * error envelopes to WP_Error.
	 *
	 * @param array|WP_Error $response Raw wp_remote_* response.
	 * @return array|WP_Error Unwrapped data or error.
	 */
	private static function parse_response( $response ) {
		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true ) ?? array();

		// Unwrap { ok, data } / { ok, error } envelope.
		if ( is_array( $body ) && isset( $body['ok'] ) ) {
			if ( false === $body['ok'] ) {
				$message = isset( $body['error']['message'] )
					? $body['error']['message']
					: 'API error (HTTP ' . $code . ')';
				return new WP_Error( 'gotasale_api_error', $message );
			}
			return isset( $body['data'] ) ? $body['data'] : array();
		}

		// Non-envelope HTTP error.
		if ( $code >= 400 ) {
			return new WP_Error( 'gotasale_api_error', 'API error (HTTP ' . $code . ')' );
		}

		return $body;
	}

	/**
	 * Make a POST request to the GotASale API.
	 *
	 * @param string $endpoint API endpoint path.
	 * @param array  $body     Request body.
	 * @return array|WP_Error Decoded response or error.
	 */
	private static function post( $endpoint, $body ) {
		$token = get_option( 'gotasale_site_token' );

		if ( ! $token ) {
			return new WP_Error( 'gotasale_no_token', __( 'Site token not configured.', 'got-a-sale' ) );
		}

		$response = wp_remote_post( GOTASALE_API_URL . $endpoint, array(
			'headers'   => array(
				'Content-Type'  => 'application/json',
				'X-Site-Token'  => $token,
			),
			'body'      => wp_json_encode( $body ),
			'timeout'   => 15,
			'sslverify' => true,
		) );

		return self::parse_response( $response );
	}

	/**
	 * Make a DELETE request to the GotASale API.
	 *
	 * @param string $endpoint API endpoint path.
	 * @return array|WP_Error Decoded response or error.
	 */
	private static function delete( $endpoint ) {
		$token = get_option( 'gotasale_site_token' );

		if ( ! $token ) {
			return new WP_Error( 'gotasale_no_token', __( 'Site token not configured.', 'got-a-sale' ) );
		}

		$response = wp_remote_request( GOTASALE_API_URL . $endpoint, array(
			'method'    => 'DELETE',
			'headers'   => array(
				'X-Site-Token' => $token,
			),
			'timeout'   => 15,
			'sslverify' => true,
		) );

		return self::parse_response( $response );
	}

	/**
	 * Make a GET request to the GotASale API.
	 *
	 * @param string $endpoint API endpoint path.
	 * @return array|WP_Error Decoded response or error.
	 */
	private static function get( $endpoint ) {
		$token = get_option( 'gotasale_site_token' );

		if ( ! $token ) {
			return new WP_Error( 'gotasale_no_token', __( 'Site token not configured.', 'got-a-sale' ) );
		}

		$response = wp_remote_get( GOTASALE_API_URL . $endpoint, array(
			'headers'   => array(
				'X-Site-Token' => $token,
			),
			'timeout'   => 15,
			'sslverify' => true,
		) );

		return self::parse_response( $response );
	}
}
