<?php
/**
 * Settings API registration.
 *
 * Registers all GotASale settings with the WordPress Settings API.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class GotASale_Settings {

	/**
	 * Register settings hooks.
	 */
	public function init() {
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	/**
	 * Register settings sections and fields.
	 */
	public function register_settings() {
		// Connection section
		add_settings_section(
			'gotasale_connection',
			__( 'Connection', 'got-a-sale' ),
			array( $this, 'render_connection_section' ),
			'got-a-sale'
		);

		// Site token (read-only display)
		add_settings_field(
			'gotasale_site_token',
			__( 'Site Token', 'got-a-sale' ),
			array( $this, 'render_site_token_field' ),
			'got-a-sale',
			'gotasale_connection'
		);

		// License key
		register_setting( 'got-a-sale', 'gotasale_license_key', array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
		) );

		add_settings_field(
			'gotasale_license_key',
			__( 'License Key', 'got-a-sale' ),
			array( $this, 'render_license_key_field' ),
			'got-a-sale',
			'gotasale_connection'
		);

		// Notification settings section
		add_settings_section(
			'gotasale_notifications',
			__( 'Notifications', 'got-a-sale' ),
			array( $this, 'render_notifications_section' ),
			'got-a-sale'
		);

		register_setting( 'got-a-sale', 'gotasale_notification_settings', array(
			'type'              => 'array',
			'sanitize_callback' => array( $this, 'sanitize_notification_settings' ),
		) );
	}

	/**
	 * Render connection section description.
	 */
	public function render_connection_section() {
		echo '<p>' . esc_html__( 'Connect your store to the Got A Sale notification service.', 'got-a-sale' ) . '</p>';
	}

	/**
	 * Render the read-only site token field with copy button.
	 */
	public function render_site_token_field() {
		$token = get_option( 'gotasale_site_token', '' );
		echo '<code id="gotasale-site-token">' . esc_html( $token ) . '</code>';
		echo ' <button type="button" class="button button-small" id="gotasale-copy-token">' . esc_html__( 'Copy', 'got-a-sale' ) . '</button>';
		echo '<p class="description">' . esc_html__( 'Paste this token to @gotasale_bot on Telegram to link your store.', 'got-a-sale' ) . '</p>';
	}

	/**
	 * Render the license key field.
	 */
	public function render_license_key_field() {
		$key = get_option( 'gotasale_license_key', '' );
		echo '<input type="text" name="gotasale_license_key" value="' . esc_attr( $key ) . '" class="regular-text" />';
		echo '<p class="description">' . esc_html__( 'Enter your license key for Pro or Agency features.', 'got-a-sale' ) . '</p>';
	}

	/**
	 * Render notifications section description.
	 */
	public function render_notifications_section() {
		echo '<p>' . esc_html__( 'Choose which events trigger notifications.', 'got-a-sale' ) . '</p>';
	}

	/**
	 * Sanitize notification settings.
	 *
	 * @param array $input Raw input.
	 * @return array Sanitized settings.
	 */
	public function sanitize_notification_settings( $input ) {
		$sanitized = array();
		$valid_keys = array( 'new_order', 'status_changed', 'low_stock' );

		foreach ( $valid_keys as $key ) {
			$sanitized[ $key ] = ! empty( $input[ $key ] );
		}

		return $sanitized;
	}
}
