<?php
/**
 * GotASale API Client for PrestaShop.
 * Handles HTTP communication with the GotASale server.
 */

class GotASaleAPI
{
    private $apiUrl;
    private $timeout = 10;

    public function __construct()
    {
        $this->apiUrl = rtrim(Configuration::get('GOTASALE_API_URL') ?: 'https://api.gotasale.io', '/');
    }

    /**
     * Register this store with the GotASale server.
     */
    public function registerStore($siteToken, $storeUrl, $storeName)
    {
        return $this->post('/stores/register', [
            'site_token' => $siteToken,
            'store_url' => $storeUrl,
            'store_name' => $storeName ?: $storeUrl,
            'platform' => 'prestashop',
        ]);
    }

    /**
     * Send a notification payload to the GotASale server.
     */
    public function sendNotification($siteToken, array $payload)
    {
        return $this->post('/notify', $payload, [
            'X-Site-Token: ' . $siteToken,
        ]);
    }

    /**
     * Get pending link requests for this store.
     */
    public function getPendingLinks($siteToken)
    {
        $response = $this->get('/stores/pending-links', [
            'X-Site-Token: ' . $siteToken,
        ]);

        if ($response && isset($response['data']['pending_links'])) {
            return $response['data']['pending_links'];
        }

        return [];
    }

    /**
     * Confirm a pending link request.
     */
    public function confirmLink($siteToken, $confirmCode)
    {
        return $this->post('/stores/confirm-link', [
            'confirm_code' => $confirmCode,
        ], [
            'X-Site-Token: ' . $siteToken,
        ]);
    }

    /**
     * Send PrestaShop Webservice API key to the server for encrypted storage.
     */
    public function sendApiKey($siteToken, $apiKey)
    {
        return $this->post('/stores/prestashop-credentials', [
            'api_key' => $apiKey,
        ], [
            'X-Site-Token: ' . $siteToken,
        ]);
    }

    /**
     * Send a test notification to all linked channels.
     */
    public function sendTest($siteToken)
    {
        return $this->post('/test', [], [
            'X-Site-Token: ' . $siteToken,
        ]);
    }

    // ── HTTP Helpers ─────────────────────────────────────────

    private function post($path, array $data, array $extraHeaders = [])
    {
        $url = $this->apiUrl . $path;

        $headers = array_merge([
            'Content-Type: application/json',
            'Accept: application/json',
            'User-Agent: GotASale-PrestaShop/1.0.0',
        ], $extraHeaders);

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_SSL_VERIFYPEER => true,
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            PrestaShopLogger::addLog(
                'GotASale API error: ' . $error,
                3,
                null,
                'GotASale',
                null,
                true
            );
            return null;
        }

        return json_decode($response, true);
    }

    private function get($path, array $extraHeaders = [])
    {
        $url = $this->apiUrl . $path;

        $headers = array_merge([
            'Accept: application/json',
            'User-Agent: GotASale-PrestaShop/1.0.0',
        ], $extraHeaders);

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_HTTPGET => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_SSL_VERIFYPEER => true,
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            PrestaShopLogger::addLog(
                'GotASale API error: ' . $error,
                3,
                null,
                'GotASale',
                null,
                true
            );
            return null;
        }

        return json_decode($response, true);
    }
}
